/*
 * routines that use the regexp stuff
 * create function regex.REGEXP(string, string) returns int;
*/

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <string.h>
#include "pcre/pcre.h"

/* this is the number of cached regular expressions held. */
#ifndef MAX_CACHED_RES
#define MAX_CACHED_RES	32
#endif

/* this structure describes a cached regular expression */
struct cached_re_str
{
  char *cre_s;				/* pattern as null-terminated string */
  int  cre_type;			/* compiled-type: extended,icase etc */
  pcre *cre_re;				/* the compiled regular expression */
  pcre_extra *extra;			/* pcre_study */
  unsigned long cre_lru;		/* lru tag */
};

static int rec = 0;			/* # of cached re's */
static struct cached_re_str rev[MAX_CACHED_RES];	/* cached re's */
static unsigned long lru;		/* system lru tag */

#define isbig5head(c)   ((((unsigned char)c) >= 0xA1) && (((unsigned char)c) <= 0xFE))
#define isbig5tail(c)   (((((unsigned char)c) >= 0x40)&& (((unsigned char)c) <= 0x7E)) \
                         || ((((unsigned char)c) >= 0xA1)&&(((unsigned char)c) <= 0xFE)))
/* attempt to compile `re' as an re, then match it against text */
/* cflags - flag to regcomp indicates case sensitivity */
static int
regex_compile_and_go(char *patt, int pattlen, char *text, int textlen)
{
  int  oldest;
  const char *error;
  char  *pp;
  char  *buf;
  unsigned char parenthese[2];
  int  delimiter;
  int  i = 0;
  int  count = 0;
  int  options = 0;
  int  study_options = 0;
  int  do_study = 0;
  int  errofoffset;
  int  offsets[99];

  /* alloc buf */
  if ((buf=malloc((pattlen>>1)+pattlen))==0)
    return -1;

  /*
   * parse the pattern
   */
  while ((i < pattlen) && isspace(patt[i])) i++;	/* strip left space */
  if (i == pattlen) return 6015;

  /* Get the delimiter and seek the end of the pattern */
  delimiter = patt[i++];
  /* Delimiter must not be alphameric or \ */
  if (isalnum(delimiter) || delimiter == '\\')
    return 6015;
  switch (delimiter)
  {
    case '{': parenthese[0]='{'; parenthese[1]='}'; count=1; break;
    case '[': parenthese[0]='['; parenthese[1]=']'; count=1; break;
    case '(': parenthese[0]='('; parenthese[1]=')'; count=1; break;
  }

  /* Copy pattern to buf with null-terminator */
  pp = buf;
  while (i < pattlen)
  {
    if (isbig5head(patt[i]) && ((i+1) < pattlen) && isbig5tail(patt[i+1]))
    {
      *pp++ = patt[i++];			/* pass big5 word */
      switch (patt[i])
      {
        case '[': case ']':
        case '{': case '}':
        case '(': case ')':
        case '!': case '$': case '*': case '+': case ',':
        case '-': case '.': case ':': case '<': case '=':
        case '>': case '?': case '^': case '\\':
          *pp++ = '\\';
      }
    }
    else if (patt[i] == '\\' && ((i+1) < pattlen))
      *pp++ = patt[i++];			/* pass the next char  */
    else if (count > 0)				/* for delimiter { */
    {
      if (patt[i] == parenthese[0])
        count++;
      else if (patt[i] == parenthese[1])
        count--;
      if (count==0)
        break;
    }
    else if (patt[i] == delimiter) break;	/* for other delimiter */
    *pp++ = patt[i++];
  }
  if (i++ == pattlen) return 6015;
  *pp = 0;

  /* Look for options after final delimiter */
  while (i < pattlen)
  {
    switch (patt[i++])
    {
      case 'i': options |= PCRE_CASELESS; break;
      case 'm': options |= PCRE_MULTILINE; break;
      case 's': options |= PCRE_DOTALL; break;
      case 'x': options |= PCRE_EXTENDED; break;
      case 'A': options |= PCRE_ANCHORED; break;
      case 'E': options |= PCRE_DOLLAR_ENDONLY; break;
      case 'S': do_study = 1; break;
      case 'U': options |= PCRE_UNGREEDY; break;
      case 'X': options |= PCRE_EXTRA; break;
      default:  break;
    }
  }
  
  /* strip right space for text */
  while ((textlen > 0) && isspace(text[textlen-1])) textlen--;

  /* find a previously compiled regular expression */
  for (i = 0; i < rec; i++)
  {
    if (rev[i].cre_s &&
        (strcmp(rev[i].cre_s, buf) == 0) && 	/* same pattern */
        (rev[i].cre_type == options))		/* same flags   */
    {
      rev[i].cre_lru = ++lru;
      return (pcre_exec(rev[i].cre_re, rev[i].extra, text, textlen,
                        0, 0, offsets, 99) >= 0);
    }
  }

  /* we didn't find it - make room in the cache for it */
  if (rec == MAX_CACHED_RES)
  {
    /* cache is full - find the oldest entry */
    for (oldest = 0, i = 1; i < rec; i++)
    {
      if (rev[i].cre_lru < rev[oldest].cre_lru)
        oldest = i;
    }
  }
  else
    oldest = rec++;

  /* if there was an old patt, then de-allocate the space it used */
  if (rev[oldest].cre_s != (char *) NULL)
  {
    for (lru = i = 0; i < rec; i++)
    {
      rev[i].cre_lru = (rev[i].cre_lru - rev[oldest].cre_lru) / 2;
      if (rev[i].cre_lru > lru)
        lru = rev[i].cre_lru;
    }
    if (rev[oldest].extra)
      free(rev[oldest].extra);
    free(rev[oldest].cre_re);

    /*
     * use malloc/free for the cre_s field because the storage has to
     * persist across transactions
     */
    free(rev[oldest].cre_s);
  }

  /* compile the pattern */
  rev[oldest].cre_re = pcre_compile(buf, options, &error, &errofoffset, NULL);
  if (rev[oldest].cre_re != NULL)
  {
    if (do_study)
      rev[oldest].extra = pcre_study(rev[oldest].cre_re, 0, &error);

    rev[oldest].cre_s = buf;		/* free it when cache full */
    rev[oldest].cre_lru = ++lru;
    rev[oldest].cre_type = options;
    return (pcre_exec(rev[oldest].cre_re, rev[oldest].extra, text, textlen,
                      0, 0, offsets, 99) >= 0);
  }
  else
  {
    /* patt didn't compile */
    fprintf(stderr, "Failed: %s at offset %d\n", error, errofoffset);
    return 6015;	/* (6015): [DBMaker] incomplete SQL statement input */
  }

  /* not reached */
  return 0;
}

#ifndef NO_UDF
#include <libudf.h>

/*
 * routines that use the regexp stuff
 * create function regex.REGEXP(string, string) returns int;
 */

#ifdef WIN32
__declspec(dllexport)
#endif
int REGEXP(int narg, VAL args[])
{
  int rc = 0;

  if (args[0].type == NULL_TYP || args[1].type == NULL_TYP)
    return _RetVal(args, args[0]);

  rc = regex_compile_and_go(args[1].u.xval,	/* the pattern */
                            args[1].len,	/* the pattern length */
                            args[0].u.xval,	/* the string to match */
                            args[0].len);	/* the string length */ 
  if ((rc!=0) && (rc!=1))
    return rc;
  args[0].u.ival = rc;
  args[0].type = INT_TYP;
  return _RetVal(args, args[0]);
}

#else /* NO_UDF */
void main(void)
{
  char buf[256]="Eddie@";
  char p1[]="/E/i";
  char p2[]="/@/";
  char p3[]="   ";
  char p4[]="{[a-z]}i";
  int rc;

  rc = regex_compile_and_go(p1, strlen(p1), buf, strlen(buf));
  printf("rc: %d\n", rc);
  rc = regex_compile_and_go(p2, strlen(p2), buf, strlen(buf));
  printf("rc: %d\n", rc);
  rc = regex_compile_and_go(p3, strlen(p3), buf, strlen(buf));
  printf("rc: %d\n", rc);
  rc = regex_compile_and_go(p4, strlen(p4), buf, strlen(buf));
  printf("rc: %d\n", rc);
}

#endif /* NO_UDF */
